% function [value] = parse_varargs(varargs, name, default, class, attributes )
%   Searches varargs for name, checks the following value for type class and attributes.
%   The class and attributes arguments are optional.  Arguments not matching the class
%	and/or attributes provided will return an error. For a list of available attributes 
%	used with numeric data types consult the MATLAB validateattributes function 
%	documentation. If the requested class is 'char' then the attributes argument can be 
%	used to specify a list available strings.  An entry not matching one of the provided 
%	string will generate an error.

function [value] = parse_varargs(varargs, name, default, varargin)
    
	% Check Number of Arguments
	narginchk(3, 5);

	% Set Default Value
	value = default;

	% Type & Attributes not required so varargin
	if nargin == 3
		type = {};
		attrib = {};
	elseif nargin == 4 
		type = varargin{1};
		attrib = {};	
	elseif ischar( varargin{2} )
		type = varargin{1};
		attrib = varargin(2);		% Single Attribute
	else
		type = varargin{1};
		attrib = varargin{2};		% Multiple Attributes
	end
	
	% Convert other data types to appropriate MATLAB class
	type = strrep(type,'string','char');
	type = strrep(type,'str','char');
	type = strrep(type,'bool','logical');
	
	% Find matching Entries
	match = strcmpi( name, varargs);
	
	% Error if multiple matches found
	if sum(match) > 1
		error(['Multiple entries for parameter ', name ]);
	elseif sum(match) < 1
		return;
	end
	
	% Index of matching parameter
	idx = find(match,1) + 1;
	
	if strcmpi(type, 'toggle')
        value = ~default;
	elseif idx > length(varargs)
		error(['No matching argument provided for parameter ',name]);
	elseif	isempty(type)
		% No Class Checking
		value = varargs{idx};
	elseif strcmpi(type,'char') && isempty( attrib)
		% Check Class - No String Check
		validateattributes( varargs{idx}, {'char'}, {})
		value = varargs{idx};
	elseif strcmpi(type,'char')
		% Check Class - Check List of Strings
		validateattributes( varargs{idx}, {'char'}, {})
		value = validatestring(varargs{idx},attrib);
	else	
		% Check Class and Attributes
		validateattributes( varargs{idx}, {type}, attrib)
		value = varargs{idx};
	end
	
end
